/*
 * Copyright (c) 1994 Paul Vojta.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * NOTE:
 *	xhdvi is based on prior work as noted in the modification history, below.
 */

/*
 * DVI previewer for X.
 *
 * Eric Cooper, CMU, September 1985.
 *
 * Code derived from dvi-imagen.c.
 *
 * Modification history:
 * 1/1986	Modified for X.10	--Bob Scheifler, MIT LCS.
 * 7/1988	Modified for X.11	--Mark Eichin, MIT
 * 12/1988	Added 'R' option, toolkit, magnifying glass
 *					--Paul Vojta, UC Berkeley.
 * 2/1989	Added tpic support	--Jeffrey Lee, U of Toronto
 * 4/1989	Modified for System V	--Donald Richardson, Clarkson Univ.
 * 3/1990	Added VMS support	--Scott Allendorf, U of Iowa
 * 7/1990	Added reflection mode	--Michael Pak, Hebrew U of Jerusalem
 * 1/1992	Added greyscale code	--Till Brychcy, Techn. Univ. Muenchen
 *					  and Lee Hetherington, MIT
 * 4/1994	Added DPS support, bounding box
 *					--Ricardo Telichevesky
 *					  and Luis Miguel Silveira, MIT RLE.
 * 5/1994       Added hypertex code     --Arthur Smith, U. of Washington
 */

#include <ctype.h>
#include <math.h>
#include "xhdvi.h"

/* Xlib and Xutil are already included */
#ifdef	TOOLKIT
#ifdef	OLD_X11_TOOLKIT
#include <X11/Atoms.h>
#else /* not OLD_X11_TOOLKIT */
#include <X11/Xatom.h>
#include <X11/StringDefs.h>
#endif /* not OLD_X11_TOOLKIT */
#include <X11/Shell.h>	/* needed for def. of XtNiconX */
#ifndef	XtSpecificationRelease
#define	XtSpecificationRelease	0
#endif
#if	XtSpecificationRelease >= 4
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/TextSrc.h>
#else	/* XtSpecificationRelease < 4 */
#define	XtPointer caddr_t
#include <X11/Command.h>
#include <X11/Xaw/Scroll.h>
#include <X11/Text.h>
#endif	/* XtSpecificationRelease */
#else	/* !TOOLKIT */
typedef	int		Position;
#define	XtPending()	XPending(DISP)
#endif	/* TOOLKIT */

#if	HAS_SIGIO
#include <signal.h>
#ifndef	FASYNC
#undef	HAS_SIGIO
#define	HAS_SIGIO 0
#endif
#endif

#ifndef	X11HEIGHT
#define	X11HEIGHT	8	/* Height of server default font */
#endif

#define	MAGBORD	1	/* border size for magnifier */

/*
 * Command line flags.
 */

#define	fore_Pixel	resource._fore_Pixel
#define	back_Pixel	resource._back_Pixel
#ifdef	TOOLKIT
extern	struct _resource	resource;
#define	brdr_Pixel	resource._brdr_Pixel
#endif	/* TOOLKIT */

static	int	pageno_correct	= 1;

#define	clip_w	mane.width    /* Screen magnitudes */
#define	clip_h	mane.height
static	Position main_sx, main_sy; /* Screen?? */
static	Position mag_sx, mag_sy, new_mag_sx, new_mag_sy;
static	Boolean	mag_moved = False;

#ifdef	TOOLKIT
void	xbarscroll(),xbarjump(),ybarscroll(),ybarjump();

static	struct {
	_Xconst	char	*label;
	_Xconst	char	*name;
	int	closure;
	int	y_pos;
	}
	command_table[] = {
		{"Quit",	"quit",		'q',		30},
		{"Back",	"back",		'b',		70},
		{"Shrink1",	"sh1",		1 << 8 | 's',	100},
		{"Shrink2",	"sh2",		2 << 8 | 's',	130},
		{"Shrink3",	"sh3",		3 << 8 | 's',	160},
		{"Shrink4",	"sh4",		4 << 8 | 's',	190},
		{"Page-10",	"prev10",	10 << 8 | 'p',	230},
		{"Page-5",	"prev5",	5 << 8 | 'p',	260},
		{"Prev",	"prev",		'p',		290},
		{"Next",	"next",		'n',		320},
		{"Page+5",	"next5",	5 << 8 | 'n',	350},
		{"Page+10",	"next10",	10 << 8 | 'n',	380},
#if	PS
		{"View PS",	"postscript",	'v',		410},
#endif
};


static	void	handle_command();

static	XtCallbackRec	command_call[] = {
	{handle_command, NULL},
	{NULL,		NULL},
};

static	Arg	command_args[] = {
	{XtNlabel,	(XtArgVal) NULL},
	{XtNx,		(XtArgVal) 0},
	{XtNcallback,	(XtArgVal) command_call},
};

static void home ARGS((Boolean));
static	void paint_x_bar();
static	void paint_y_bar();
static	void scrollmane ARGS((int, int));

void
create_buttons()
{
	int i;

	for (i = 0; i < XtNumber(command_table); ++i) {
	    command_args[0].value = (XtArgVal) command_table[i].label;
	    command_args[1].value = (XtArgVal) command_table[i].y_pos;
	    command_call[0].closure = (XtPointer) &command_table[i].closure;
	    (void) XtCreateManagedWidget(command_table[i].name,
		commandWidgetClass, menu_widget,
		command_args, XtNumber(command_args));
	}
}

/* Coordinate conversion routines: */
/* From screen coordinates to pixels on the page */
screen_to_page(windowrec, s_x, s_y, page, p_x, p_y)
struct WindowRec *windowrec;
int s_x, s_y, *page, *p_x, *p_y;
{
	int cur_page;

	*page = current_page;
	*p_x = s_x + windowrec->base_ax;
	*p_y = s_y + windowrec->base_ay;
	if (xscroll_pages) {
		if (windowrec == &mane) cur_page = page_w;
		else cur_page = page_w*mane.shrinkfactor;
		*page = *p_x/cur_page;
		*p_x -= (*page)*cur_page;
	}
	if (yscroll_pages) {
		if (windowrec == &mane) cur_page = page_h;
		else cur_page = page_h*mane.shrinkfactor;
		*page = *p_y/cur_page; /* May need shrinkfactor? */
		*p_y -= (*page)*cur_page;
	}
}

/* From pixels on page to screen coordinates */
page_to_screen(windowrec, page, p_x, p_y, s_x, s_y)
struct WindowRec *windowrec;
int page, p_x, p_y, *s_x, *s_y;
{
	int cur_page;

	*s_x = p_x - windowrec->base_ax;
	*s_y = p_y - windowrec->base_ay;
	if (xscroll_pages) {
		if (windowrec == &mane) cur_page = page_w;
		else cur_page = page_w*mane.shrinkfactor;
		*s_x += page*cur_page;
	}
	if (yscroll_pages) {
		if (windowrec == &mane) cur_page = page_h;
		else cur_page = page_h*mane.shrinkfactor;
		*s_y += page*cur_page;
	}
}

/* From pixels on page to abslute coordinates */
page_to_abs(windowrec, page, p_x, p_y, a_x, a_y)
struct WindowRec *windowrec;
int page, p_x, p_y, *a_x, *a_y;
{
	int cur_page;

	*a_x = p_x;
	*a_y = p_y;
	if (xscroll_pages) {
		cur_page = unshrunk_page_w/windowrec->shrinkfactor;
		*a_x += page*cur_page;
	}
	if (yscroll_pages) {
		cur_page = unshrunk_page_h/windowrec->shrinkfactor;
		*a_y += page*cur_page;
	}
}

/* Absolute coordinates: from screen coords */
screen_to_abs(windowrec, s_x, s_y, a_x, a_y)
struct WindowRec *windowrec;
int s_x, s_y, *a_x, *a_y;
{
	*a_x = s_x + windowrec->base_ax;
	*a_y = s_y + windowrec->base_ay;
}

/* Absolute coordinates: to screen coords */
abs_to_screen(windowrec, a_x, a_y, s_x, s_y)
struct WindowRec *windowrec;
int a_x, a_y, *s_x, *s_y;
{
	*s_x = a_x - windowrec->base_ax;
	*s_y = a_y - windowrec->base_ay;
}

void xbarscroll(sb, cd, pos)
Widget sb;
caddr_t cd;
caddr_t pos;
{
	int step = 20;
	int xpos;

	xpos = (int) pos;
	if (step > 1 + clip_w/2) step = 1 + clip_w/2;
	if (xpos > 0) scrollmane(mane.base_ax - step, mane.base_ay);
	else scrollmane(mane.base_ax + step, mane.base_ay);
}

void xbarjump(sb, cd, percent)
Widget sb;
caddr_t cd;
caddr_t percent;
{
	double frac;
	int new_base;

	frac = *(float *) percent;
	new_base = abs_w * frac;
	if (mane.base_ax != new_base) {
		scrollmane(new_base, mane.base_ay);
	}
}

void ybarscroll(sb, cd, pos)
Widget sb;
caddr_t cd;
caddr_t pos;
{
	int ypos, step = 20;

	ypos = (int) pos;
	if (step > 1 + clip_h/2) step = 1 + clip_h/2;
	if (ypos > 0) scrollmane(mane.base_ax, mane.base_ay - step);
	else scrollmane(mane.base_ax, mane.base_ay + step);
}

void ybarjump(sb, cd, percent)
Widget sb;
caddr_t cd;
caddr_t percent;
{
	double frac;
	int new_base;

	frac = *(float *) percent;
	new_base = abs_h * frac;
	if (mane.base_ay != new_base) {
		scrollmane(mane.base_ax, new_base);
	}
}


#else	/* !TOOLKIT */
static	Window	x_bar, y_bar;
static	int	sx_bgn, sx_end, sy_bgn, sy_end;	/* scrollbar positions (scrn) */
#endif	/* TOOLKIT */

/*
 *	Mechanism to keep track of the magnifier window.  The problems are,
 *	(a) if the button is released while the window is being drawn, this
 *	could cause an X error if we continue drawing in it after it is
 *	destroyed, and
 *	(b) creating and destroying the window too quickly confuses the window
 *	manager, which is avoided by waiting for an expose event before
 *	destroying it.
 */
static	short	alt_stat;	/* 1 = wait for expose, */
				/* -1 = destroy upon expose */
static	Boolean	alt_canit;	/* stop drawing this window */

/*
 *	Data for buffered events.
 */

static	Boolean	has_arg		= False;
static	VOLATILE short	event_freq	= 70;
static	int	number		= 0;
static	int	sign		= 1;

static	void	can_exposures(), keystroke();

#ifdef	GREY
#define	gamma	resource._gamma

void
init_pix(warn)
	Boolean	warn;
{
	static	int	shrink_allocated_for = 0;
	static	Boolean	colors_allocated = False;
	int	i;

	if (!colors_allocated)
	{
	    Pixel plane_masks[4];
	    Pixel pixel;
	    XColor color, fc, bc;
	    XGCValues	values;

	    if (gamma == 0.0) gamma = 1.0;

	    if (!resource.copy)
		/* allocate 4 color planes for 16 colors (for GXor drawing) */
		if (!XAllocColorCells(DISP, DefaultColormapOfScreen(SCRN),
					  False, plane_masks, 4, &pixel, 1))
		    resource.copy = warn = True;

	    /* get foreground and background RGB values for interpolating */
	    fc.pixel = fore_Pixel;
	    XQueryColor(DISP, DefaultColormapOfScreen(SCRN), &fc);
	    bc.pixel = back_Pixel;
	    XQueryColor(DISP, DefaultColormapOfScreen(SCRN), &bc);

	    for (i = 0; i < 16; ++i) {
		double	frac = gamma > 0 ? pow((double) i / 15, 1 / gamma)
		    : 1 - pow((double) (15 - i) / 15, -gamma);

		color.red = frac * ((double) fc.red - bc.red) + bc.red;
		color.green = frac * ((double) fc.green - bc.green) + bc.green;
		color.blue = frac * ((double) fc.blue - bc.blue) + bc.blue;

		color.pixel = pixel;
		color.flags = DoRed | DoGreen | DoBlue;

		if (!resource.copy) {
		    if (i & 1) color.pixel |= plane_masks[0];
		    if (i & 2) color.pixel |= plane_masks[1];
		    if (i & 4) color.pixel |= plane_masks[2];
		    if (i & 8) color.pixel |= plane_masks[3];
		    XStoreColor(DISP, DefaultColormapOfScreen(SCRN), &color);
		    palette[i] = color.pixel;
		}
		else {
		    if (!XAllocColor(DISP, DefaultColormapOfScreen(SCRN),
			&color))
			palette[i] = (i * 100 >= density * 15)
			    ? fore_Pixel : back_Pixel;
		    else
			palette[i] = color.pixel;
		}
	    }

	    /* Make sure fore_ and back_Pixel are a part of the palette */
	    fore_Pixel = palette[15];
	    back_Pixel = palette[0];
	    if (mane.win != (Window) 0)
		XSetWindowBackground(DISP, mane.win, palette[0]);

#define	MakeGC(fcn, fg, bg)	(values.function = fcn, values.foreground=fg,\
		values.background=bg,\
		XCreateGC(DISP, RootWindowOfScreen(SCRN),\
			GCFunction|GCForeground|GCBackground, &values))

	    foreGC = ruleGC = MakeGC(resource.copy ? GXcopy : GXor,
		fore_Pixel, back_Pixel);
	    foreGC2 = NULL;

	    colors_allocated = True;
	    if (resource.copy && warn)
		Puts("Note:  overstrike characters may be incorrect.");
	}
#undef	MakeGC

	if (mane.shrinkfactor == 1) return;

	if (shrink_allocated_for < mane.shrinkfactor) {
	    if (pixeltbl != NULL) free((char *) pixeltbl);
	    pixeltbl = (Pixel *) xmalloc((unsigned)
		(mane.shrinkfactor * mane.shrinkfactor + 1) * sizeof(Pixel),
		"pixel table");
	    shrink_allocated_for = mane.shrinkfactor;
	}

	for (i = 0; i <= mane.shrinkfactor * mane.shrinkfactor; ++i)
	    pixeltbl[i] =
		palette[(i * 30 + mane.shrinkfactor * mane.shrinkfactor)
		    / (2 * mane.shrinkfactor * mane.shrinkfactor)];
}
#endif	/* GREY */

#if	defined(sun) && BSD
extern	char	*sprintf();
#endif

#ifndef	atof	/* on some machines it's a macro */
extern	double	atof();
#endif

/*
 *	Event-handling routines
 */

static	void
expose(windowrec, sx, sy, w, h)
	register struct WindowRec *windowrec;
	int		sx, sy;
	unsigned int	w, h;
{
	if (windowrec->min_sx > sx) windowrec->min_sx = sx;
	if (windowrec->max_sx < sx + w)
	    windowrec->max_sx = sx + w;
	if (windowrec->min_sy > sy) windowrec->min_sy = sy;
	if (windowrec->max_sy < sy + h)
	    windowrec->max_sy = sy + h;
}

static	void
clearexpose(windowrec, sx, sy, w, h)
	struct WindowRec *windowrec;
	int		sx, sy;
	unsigned int	w, h;
{
	XClearArea(DISP, windowrec->win, sx, sy, w, h, False);
	expose(windowrec, sx, sy, w, h);
}

/* ax0, ay0 are absolute locators: include page number if
   multiple pages on one screen */

static	void
scrollwindow(windowrec, ax0, ay0)
	register struct WindowRec *windowrec;
	int	ax0, ay0;
{
	int	sx, sy;
	int	sx2 = 0, sy2 = 0;
	int	ww, hh;

	abs_to_screen(windowrec, ax0, ay0, &sx, &sy); /* Convert to screen coords */
	ww = windowrec->width - sx;
	hh = windowrec->height - sy;
	windowrec->base_ax = ax0; /* Changing window position! */
	windowrec->base_ay = ay0;
	if (currwin.win == windowrec->win) {
	    currwin.base_ax = ax0;
	    currwin.base_ay = ay0;
	}
	windowrec->min_sx -= sx;
	if (windowrec->min_sx < 0) windowrec->min_sx = 0;
	windowrec->max_sx -= sx;
	if (windowrec->max_sx > windowrec->width)
	    windowrec->max_sx = windowrec->width;
	windowrec->min_sy -= sy;
	if (windowrec->min_sy < 0) windowrec->min_sy = 0;
	windowrec->max_sy -= sy;
	if (windowrec->max_sy > windowrec->height)
	    windowrec->max_sy = windowrec->height;
	if (sx < 0) {
	    sx2 = -sx;
	    sx = 0;
	    ww = windowrec->width - sx2;
	}
	if (sy < 0) {
	    sy2 = -sy;
	    sy = 0;
	    hh = windowrec->height - sy2;
	}
	if (ww <= 0 || hh <= 0) {
	    XClearWindow(DISP, windowrec->win);
	    windowrec->min_sx = windowrec->min_sy = 0;
	    windowrec->max_sx = windowrec->width;
	    windowrec->max_sy = windowrec->height;
	}
	else {
	    XCopyArea(DISP, windowrec->win, windowrec->win,
		DefaultGCOfScreen(SCRN), sx, sy,
		(unsigned int) ww, (unsigned int) hh, sx2, sy2);
	    if (sx > 0)
		clearexpose(windowrec, ww, 0,
		    (unsigned int) sx, windowrec->height);
	    if (sx2 > 0)
		clearexpose(windowrec, 0, 0,
		    (unsigned int) sx2, windowrec->height);
	    if (sy > 0)
		clearexpose(windowrec, 0, hh,
		    windowrec->width, (unsigned int) sy);
	    if (sy2 > 0)
		clearexpose(windowrec, 0, 0,
		    windowrec->width, (unsigned int) sy2);
	}
}

#ifdef	TOOLKIT
/*
 *	routines for X11 toolkit
 */

static	Arg	arg_wh[] = {
	{XtNwidth,	(XtArgVal) &window_w},
	{XtNheight,	(XtArgVal) &window_h},
};

static	Arg	arg_xy[] = {
	{XtNx,	(XtArgVal) &window_w},
	{XtNy,	(XtArgVal) &window_h},
};

static	Boolean	resized	= False;

/* Get screen dimensions and set things up appropriately */
static	void
get_geom()
{
	XtGetValues(draw_widget, arg_wh, XtNumber(arg_wh));

	clip_w = window_w;
	clip_h = window_h;
	paint_x_bar();
	paint_y_bar();
	resized = False;
}

/*
 *	callback routines
 */

/* The following callback routine should never be called. */
	/*ARGSUSED*/
void
handle_key(widget, junk, event, cont)
	Widget	widget;
	XtPointer junk;
	XEvent	*event;
	Boolean	*cont;		/* unused */
{
	XBell(DISP, 20);
}

	/*ARGSUSED*/
void
handle_resize(widget, junk, event, cont)
	Widget	widget;
	XtPointer junk;
	XEvent	*event;
	Boolean	*cont;		/* unused */
{
	resized = True;
}

	/*ARGSUSED*/
static	void
handle_command(widget, client_data_p, call_data)
	Widget	widget;
	XtPointer client_data_p;
	XtPointer call_data;
{
	int	client_data	= * (int *) client_data_p;

	keystroke((client_data) & 0xff, (client_data) >> 8,
		((client_data) >> 8) != 0, (XEvent *) NULL);
}

void
reconfig()
{
	get_geom();
}

#define BIGBUF 32768
char anchorpaintstring[BIGBUF];
static Arg new_paint_args[] = {
	{XtNstring,	(XtArgVal) anchorpaintstring},
};

/* Using Text widget ??? - see if this works... */
void paint_anchor(text)
char *text;
{
	static long pos = 0; /* Keep track of how many characters we've added */
	int len;
	char *firstnl;

	if (text == NULL) { /* Not on anchor any more */
		if ((pos > 0) && (pos < BIGBUF) && (anchorpaintstring[pos-1] != '\n')) {
			anchorpaintstring[pos]= '\n'; /* But we were */
			anchorpaintstring[pos+1]= '\0';
			pos += 1;
			XtSetValues(anchor_info, new_paint_args,
				XtNumber(new_paint_args));
			XawTextSetInsertionPoint(anchor_info, pos);
		}
		return;
	} /* There IS some anchor text under the mouse */
	len = strlen(text);
	if (!strcmp(text, anchorpaintstring+pos-len)) { /* Still on same anchor */
		return;
	}
	if (len > BIGBUF) { /* Try it anyway: */
		text[BIGBUF-1] = '\0';
		len = BIGBUF;
	}
	if ((pos + len) > BIGBUF) { /* Fix in case of overflows... */
		firstnl = index(anchorpaintstring+pos+len-BIGBUF, '\n');
		if (firstnl == NULL) {
			pos = 0;
		} else {
			strcpy(anchorpaintstring,firstnl+1);
			pos -= 1 + firstnl - anchorpaintstring;
		}
	}
	if (pos < 0) pos = 0; /* Shouldn't happen but... */
	sprintf(anchorpaintstring+pos,"\n%s", text);
	pos += len+1;

	XtSetValues(anchor_info, new_paint_args, XtNumber(new_paint_args));
	XawTextSetInsertionPoint(anchor_info, pos);
	XawTextDisplay(anchor_info);
}

extern char anchorsearchstring[]; /* not being used? */

/* Callback for anchor search: */
void search_callback(w, c_p, call_data)
Widget w;
XtPointer c_p, call_data;
{
	int len;
	char *cp;

	cp = XawDialogGetValueString(anchor_search);
	if (cp == NULL) return;
	len = strlen(cp);
	strcpy(anchorsearchstring,cp);
	if (index(cp,'\n') == NULL) {
		anchorsearchstring[len] = '\n';
		anchorsearchstring[len+1] = '\0';
	}
	len = index(anchorsearchstring,'\n') - anchorsearchstring + 1;
	add_search(anchorsearchstring, len);
}

#else	/* !TOOLKIT */

/* The anchor window stuff: */

void paint_anchor(text)
char *text;
{
	ClearArea(anchor_info, 1, 1, clip_w/2, BAR_THICK);
	if (text != NULL) XDrawString(DISP, anchor_info, foreGC,
		1 , BAR_WID, text, strlen(text));
}

void paint_search(text)
char *text;
{
	ClearArea(anchor_search, 1, 1, clip_w/2, BAR_THICK);
	if (text != NULL) XDrawString(DISP, anchor_search, foreGC,
		1 , BAR_WID, text, strlen(text));
}

void
reconfig()
{
	int	x_thick, x_thick0 = 0;
	int	y_thick = 0;

		/* determine existence of scrollbars */
	x_thick0 = x_thick = BAR_THICK; /* APS --- Allow for anchor window */
	if (clip_w < abs_w) x_thick = 2*BAR_THICK;
	if (clip_h - x_thick < abs_h) y_thick = BAR_THICK;
	clip_w = clip_w - y_thick;
	if (clip_w < abs_w) x_thick = 2*BAR_THICK;
	clip_h = clip_h - x_thick;

		/* process drawing (clip) window */
	if (mane.win == (Window) 0) {	/* initial creation */
	    XWindowAttributes attrs;

	    mane.win = XCreateSimpleWindow(DISP, top_level, y_thick, x_thick,
			(unsigned int) clip_w, (unsigned int) clip_h, 0,
			brdr_Pixel, back_Pixel);
/* APS Notice motion of the pointer to display HTeX anchors */
	    XSelectInput(DISP, mane.win, ExposureMask |
			PointerMotionMask | PointerMotionHintMask |
			ButtonPressMask | ButtonReleaseMask);
	    (void) XGetWindowAttributes(DISP, mane.win, &attrs);
	    backing_store = attrs.backing_store;
	    XMapWindow(DISP, mane.win);
	}
	else
	    XMoveResizeWindow(DISP, mane.win, y_thick, x_thick, clip_w, clip_h);

	/* APS --- process anchor window */
	if (anchor_info) {
		XMoveResizeWindow(DPY anchor_info,
		    y_thick - 1, x_thick - BAR_THICK - 1,
					 clip_w/2, BAR_THICK - 1);
		paint_anchor(NULL);
		XMoveResizeWindow(DPY anchor_search,
		    y_thick + clip_w/2 - 1, x_thick - BAR_THICK - 1,
						 clip_w/2, BAR_THICK - 1);
	} else {
		anchor_info = XCreateSimpleWindow(DISP, top_level, y_thick - 1,
				x_thick - BAR_THICK -1,
				(unsigned int) clip_w/2, BAR_THICK - 1, 1,
				brdr_Pixel, back_Pixel);
		XSelectInput(DISP, anchor_info, ExposureMask);
		anchor_search = XCreateSimpleWindow(DISP, top_level,
				y_thick + clip_w/2 - 1,
				x_thick - BAR_THICK -1,
				(unsigned int) clip_w/2, BAR_THICK - 1, 1,
				brdr_Pixel, back_Pixel);
		XSelectInput(DISP, anchor_search, ExposureMask|KeyPressMask);
		XMapWindow(DPY anchor_info);
		XMapWindow(DPY anchor_search);
	}

		/* process scroll bars */
	if (x_thick) {
	    if (x_bar) {
		XMoveResizeWindow(DISP, x_bar,
		    y_thick - 1, -1, clip_w, BAR_THICK - 1);
		paint_x_bar();
	    }
	    else {
		x_bar = XCreateSimpleWindow(DISP, top_level, y_thick - 1, -1,
				(unsigned int) clip_w, BAR_THICK - 1, 1,
				brdr_Pixel, back_Pixel);
		XSelectInput(DISP, x_bar,
			ExposureMask | ButtonPressMask | Button2MotionMask);
		XMapWindow(DISP, x_bar);
	    }
	    sx_bgn = mane.base_ax * clip_w / abs_w;
	    sx_end = (mane.base_ax + clip_w) * clip_w / abs_w;
	}
	else
	    if (x_bar) {
		XDestroyWindow(DISP, x_bar);
		x_bar = (Window) 0;
	    }

	if (y_thick) {
	    if (y_bar) {
		XMoveResizeWindow(DISP, y_bar,
		    -1, x_thick - 1, BAR_THICK - 1, clip_h);
		paint_y_bar();
	    }
	    else {
		y_bar = XCreateSimpleWindow(DISP, top_level, -1, x_thick - 1,
				BAR_THICK - 1, (unsigned int) clip_h, 1,
				brdr_Pixel, back_Pixel);
		XSelectInput(DISP, y_bar,
			ExposureMask | ButtonPressMask | Button2MotionMask);
		XMapWindow(DISP, y_bar);
	    }
	    sy_bgn = mane.base_ay * clip_h / abs_h;
	    sy_end = (mane.base_ay + clip_h) * clip_h / abs_h;
	}
	else
	    if (y_bar) {
		XDestroyWindow(DISP, y_bar);
		y_bar = (Window) 0;
	    }
}

#endif	/* TOOLKIT */

/* Not sure of the purpose of this routine ... */
static	void
home(scrl)
	Boolean	scrl;
{
	int	px = 0, py = 0;
	int	page, sx, sy, ax, ay;

	if (page_w > clip_w) {
	    px = (page_w - clip_w) / 2;
	    if (px > home_x / mane.shrinkfactor)
		px = home_x / mane.shrinkfactor;
	}
	if (page_h > clip_h) {
	    py = (page_h - clip_h) / 2;
	    if (py > home_y / mane.shrinkfactor)
		py = home_y / mane.shrinkfactor;
	}
	if (xscroll_pages) {
		page = mane.base_ax/page_w;
	} else if (yscroll_pages) {
		page = mane.base_ay/page_h;
	} else {
		page = current_page;
	}
	page_to_screen(&mane, page, px, py, &sx, &sy);
	screen_to_abs(&mane, sx, sy, &ax, &ay);
	if (scrl)
	    scrollmane(ax, ay);
	else {
	    mane.base_ax = ax;
	    mane.base_ay = ay;
	    if (currwin.win == mane.win) {
		currwin.base_ax = ax;
		currwin.base_ay = ay;
	    }
	    if (x_bar) paint_x_bar();
	    if (y_bar) paint_y_bar();
	}
}

/*
 *	scrollbar routines: somewhat brute force with or without TOOLKIT
 */

static	void
paint_x_bar()
{
#ifdef TOOLKIT
	float sx_top = ((double) mane.base_ax)/((double) abs_w);
	float bar_len = ((double) clip_w)/((double) abs_w);

#if	XtSpecificationRelease >= 4
		XawScrollbarSetThumb(x_bar, sx_top, bar_len);
#else
		XtScrollbarSetThumb(x_bar, sx_top, bar_len);
#endif

#else
	register int	new_sx_bgn = mane.base_ax * clip_w / abs_w;
	register int	new_sx_end = (mane.base_ax + clip_w) * clip_w / abs_w;

	if (new_sx_bgn >= sx_end || sx_bgn >= new_sx_end) {	/* no overlap */
	    XClearArea(DISP, x_bar, sx_bgn, 1, sx_end - sx_bgn, BAR_WID, False);
	    XFillRectangle(DISP, x_bar, ruleGC,
		new_sx_bgn, 1, new_sx_end - new_sx_bgn, BAR_WID);
	}
	else {		/* this stuff avoids flicker */
	    if (sx_bgn < new_sx_bgn)
		XClearArea(DISP, x_bar, sx_bgn, 1, new_sx_bgn - sx_bgn,
		    BAR_WID, False);
	    else
		XFillRectangle(DISP, x_bar, ruleGC,
		    new_sx_bgn, 1, sx_bgn - new_sx_bgn, BAR_WID);
	    if (new_sx_end < sx_end)
		XClearArea(DISP, sx_bar, new_sx_end, 1, sx_end - new_sx_end,
		    BAR_WID, False);
	    else
		XFillRectangle(DISP, x_bar, ruleGC,
		    sx_end, 1, new_sx_end - sx_end, BAR_WID);
	}
	sx_bgn = new_sx_bgn;
	sx_end = new_sx_end;
#endif
}

static	void
paint_y_bar()
{
#ifdef TOOLKIT
	float sy_top = ((double) mane.base_ay)/((double) abs_h);
	float bar_len = ((double) clip_h)/((double) abs_h);

#if	XtSpecificationRelease >= 4
	XawScrollbarSetThumb(y_bar, sy_top, bar_len);
#else
	XtScrollbarSetThumb(y_bar, sy_top, bar_len);
#endif /* XtSpecificationRelease */
#else /* Toolkit */
	register int	new_sy_bgn = mane.base_ay * clip_h / abs_h;
	register int	new_sy_end = (mane.base_ay + clip_h) * clip_h / abs_h;

	if (new_sy_bgn >= sy_end || sy_bgn >= new_sy_end) {	/* no overlap */
	    XClearArea(DISP, y_bar, 1, sy_bgn, BAR_WID, sy_end - sy_bgn, False);
	    XFillRectangle(DISP, y_bar, ruleGC,
		1, new_sy_bgn, BAR_WID, new_sy_end - new_sy_bgn);
	}
	else {		/* this stuff avoids flicker */
	    if (sy_bgn < new_sy_bgn)
		XClearArea(DISP, y_bar, 1, sy_bgn, BAR_WID, new_sy_bgn - sy_bgn,
		    False);
	    else
		XFillRectangle(DISP, y_bar, ruleGC,
		    1, new_sy_bgn, BAR_WID, sy_bgn - new_sy_bgn);
	    if (new_sy_end < sy_end)
		XClearArea(DISP, y_bar, 1, new_sy_end,
		    BAR_WID, sy_end - new_sy_end, False);
	    else
		XFillRectangle(DISP, y_bar, ruleGC,
		    1, sy_end, BAR_WID, new_sy_end - sy_end);
	}
	sy_bgn = new_sy_bgn;
	sy_end = new_sy_end;
#endif /* Toolkit */
}


/* APS Pointer locator: */

int pointerlocate(xpos, ypos) /* Return screen positions */
int *xpos, *ypos;
{
	Window root, child;
	int root_x, root_y;
	unsigned int keys_buttons;

	return XQueryPointer(DISP, mane.win, &root, &child,
			&root_x, &root_y, xpos, ypos, &keys_buttons);
}

static	void
scrollmane(ax, ay) /* Absolute positions */
	int	ax, ay;
{
	register int	old_base_ax = mane.base_ax;
	register int	old_base_ay = mane.base_ay;

#if	PS
	psp.interrupt();
#endif
	if (ax > (int) (abs_w - clip_w)) ax = abs_w - clip_w;
	if (ax < 0) ax = 0;
	if (ay > (int) (abs_h - clip_h)) ay = abs_h - clip_h;
	if (ay < 0) ay = 0;
	scrollwindow(&mane, ax, ay);
	if (old_base_ax != mane.base_ax && x_bar) paint_x_bar();
	if (old_base_ay != mane.base_ay && y_bar) paint_y_bar();
}

static	void
compute_mag_pos(xp, yp) /* Screen positions */
	int	*xp, *yp;
{
	register int t;

	t = mag_sx + main_sx - alt.width/2;
	if (t > WidthOfScreen(SCRN) - (int) alt.width - 2*MAGBORD)
	    t = WidthOfScreen(SCRN) - (int) alt.width - 2*MAGBORD;
	if (t < 0) t = 0;
	*xp = t;
	t = mag_sy + main_sy - alt.height/2;
	if (t > HeightOfScreen(SCRN) - (int) alt.height - 2*MAGBORD)
	    t = HeightOfScreen(SCRN) - (int) alt.height - 2*MAGBORD;
	if (t < 0) t = 0;
	*yp = t;
}

#ifdef	TOOLKIT
	/*ARGSUSED*/
void
handle_button(widget, junk, ev, cont)
	Widget	widget;
	XtPointer junk;
	XEvent *ev;
#define	event	(&(ev->xbutton))
	Boolean	*cont;		/* unused */
#else	/* !TOOLKIT */
void
handle_button(event)
	XButtonEvent *event;
#endif	/* TOOLKIT */
{
	int s_x, s_y, ret;
	int	px, py, page;
	struct mg_size_rec	*size_ptr = mg_size + event->button - 1;
	XSetWindowAttributes attr;

	if ((event->button == 1)||(event->button == 2)) {
		if (pointerlocate(&s_x, &s_y)) {
	/* Only do this if there's actually an href right there */
			screen_to_page(&mane,s_x,s_y,&page,&px,&py);
			if (event->button == 2) HTeXnext_extern = 1;
			ret = htex_handleref(page, px, py);
			HTeXnext_extern = 0;
			if (ret == 1) return;
		}
	}
	if (alt.win != (Window) 0 || mane.shrinkfactor == 1 || size_ptr->w <= 0)
	    XBell(DISP, 20);
	else {
	    mag_sx = event->x;
	    mag_sy = event->y;
	    alt.width = size_ptr->w;
	    alt.height = size_ptr->h;
	    main_sx = event->x_root - mag_sx;
	    main_sy = event->y_root - mag_sy;
	    compute_mag_pos(&s_x, &s_y);
	    /* Should probably call screen_to_abs on this: */
	    alt.base_ax = (event->x + mane.base_ax) * mane.shrinkfactor -
		alt.width/2;
	    alt.base_ay = (event->y + mane.base_ay) * mane.shrinkfactor -
		alt.height/2;
	    attr.save_under = True;
	    attr.border_pixel = brdr_Pixel;
	    attr.background_pixel = back_Pixel;
	    attr.override_redirect = True;
	    alt.win = XCreateWindow(DISP, RootWindowOfScreen(SCRN),
			s_x, s_y, alt.width, alt.height, MAGBORD,
			0,	/* depth from parent */
			InputOutput, (Visual *) CopyFromParent,
			CWSaveUnder | CWBorderPixel | CWBackPixel |
			CWOverrideRedirect, &attr);
	    XSelectInput(DISP, alt.win, ExposureMask);
	    XMapWindow(DISP, alt.win);
	    alt_stat = 1;	/* waiting for exposure */
	}
}

#ifdef	TOOLKIT
#undef	event

	/*ARGSUSED*/
void
handle_motion(widget, junk, ev, cont)
	Widget	widget;
	XtPointer junk;
	XEvent *ev;
#define	event	(&(ev->xmotion))
	Boolean	*cont;		/* unused */
{
	int s_x, s_y;
	int page, px, py;

/* APS Modification required because of changes in motion events selected: */
	/* We asked only for hints, so check where pointer is: */
	if (pointerlocate(&s_x, &s_y)) {
		screen_to_page(&mane,s_x,s_y,&page,&px,&py);
		if ((event->state&(Button1Mask|Button2Mask|Button3Mask|
					Button4Mask|Button5Mask)) == 0) {
			htex_displayanchor(page, px, py);
		} else {
			new_mag_sx = s_x;
			main_sx = event->x_root - s_x;
			new_mag_sy = s_y;
			main_sy = event->y_root - s_y;
			mag_moved = (new_mag_sx != mag_sx || new_mag_sy != mag_sy);
		}
	}
}

#undef	event
#endif	/* TOOLKIT */

static	void
movemag(sx, sy)
	int	sx, sy;
{
	int	xx, yy;

	mag_sx = sx;
	mag_sy = sy;
	if (mag_sx == new_mag_sx && mag_sy == new_mag_sy) mag_moved = False;
	compute_mag_pos(&xx, &yy);
	XMoveWindow(DISP, alt.win, xx, yy);
	scrollwindow(&alt,
	    /* Should probably call screen_to_abs on this: */
	    (sx + mane.base_ax) * mane.shrinkfactor - (int) alt.width/2,
	    (sy + mane.base_ay) * mane.shrinkfactor - (int) alt.height/2);
}

#ifdef	TOOLKIT
	/*ARGSUSED*/
void
handle_release(widget, junk, ev, cont)
	Widget	widget;
	XtPointer junk;
	XEvent *ev;
#define	event	(&(ev->xbutton))
	Boolean	*cont;		/* unused */
#else	/* !TOOLKIT */
void
handle_release()
#endif	/* TOOLKIT */
{
	if (alt.win != (Window) 0)
	    if (alt_stat) alt_stat = -1;	/* destroy upon expose */
	    else {
		XDestroyWindow(DISP, alt.win);
		if (currwin.win == alt.win) alt_canit = True;
		alt.win = (Window) 0;
		mag_moved = False;
		can_exposures(&alt);
	    }
}

#ifdef	TOOLKIT
#undef	event

	/*ARGSUSED*/
void
handle_exp(widget, closure, ev, cont)
	Widget	widget;
	XtPointer closure;
	register XEvent *ev;
#define	event	(&(ev->xexpose))
	Boolean	*cont;		/* unused */
{
	struct WindowRec *windowrec = (struct WindowRec *) closure;

	if (windowrec == &alt)
	    if (alt_stat < 0) {	/* destroy upon exposure */
		alt_stat = 0;
		handle_release(widget, (caddr_t) NULL, ev, (Boolean *) NULL);
		return;
	    }
	    else
		alt_stat = 0;
	expose(windowrec, event->x, event->y,
	    (unsigned int) event->width, (unsigned int) event->height);
}

#undef	event
#endif	/* TOOLKIT */

/* |||
 *	Currently the event handler does not coordinate XCopyArea requests
 *	with GraphicsExpose events.  This can lead to problems if the window
 *	is partially obscured and one, for example, drags a scrollbar.
 */

static	void
keystroke(ch, number0, arg0, eventp)
	char	ch;
	int	number0;
	Boolean	arg0;
	XEvent	*eventp;
{
	int	next_page;
#ifdef	TOOLKIT
	Window	ww;
#endif
/* APS Pointer motion stuff: */
	int	s_x, s_y, page, px, py, ax, ay;

	if (xscroll_pages) {
		current_page = mane.base_ax/page_w; /* Actually "base" page */
	} else if (yscroll_pages) {
		current_page = mane.base_ay/page_h;
	}
	next_page = current_page;
	switch (ch) {
	    case 'q':
	    case '\003':	/* control-C */
	    case '\004':	/* control-D */
#ifdef	VMS
	    case '\032':	/* control-Z */
#endif
#if	PS
		psp.destroy();
#endif
		cleanup_and_exit(0);

	    case 'n':
	    case 'N':
	    case '\r':
	    case '\n':
		/* scroll forward; i.e. go to relative page */
		next_page = current_page + (arg0 ? number0 : 1);
		break;
	    case 'F':
	    case 'f': /* Follow link forward! */
		if (pointerlocate(&s_x, &s_y)) {
			screen_to_page(&mane,s_x,s_y,&page,&px,&py);
			(void) htex_handleref(page, px, py);
		}
		return; /* Should goto bad if problem arises? */
	    case 'b':
	    case 'B': /* Go back to previous anchor. */
		htex_goback(); /* Should goto bad if problem arises? */
		return;
	    case 'p':
	    case '\b':
	    case '\177':	/* Del */
		/* scroll backward */
		next_page = current_page - (arg0 ? number0 : 1);
		break;
	    case 'g':
		/* go to absolute page */
		next_page = (arg0 ? number0 - pageno_correct :
		    total_pages - 1);
		break;
	    case 'P':		/* declare current page */
		pageno_correct = arg0 * number0 - current_page;
		return;
	    case 'k':		/* toggle keep-position flag */
		resource.keep_flag = (arg0 ? number0 : !resource.keep_flag);
		return;
	    case '\f':
		/* redisplay current page */
		break;
	    case '^':
		home(True);
		return;
	    case ' ':
                /* scroll forward; first down page, then top of next page */
                if (mane.base_ay >= (int)(abs_h - clip_h)) {
                    next_page = current_page + 1;
		    home(False);
                    break;
                } else {
                    scrollmane(mane.base_ax, mane.base_ay + 2 * (int) clip_h / 3);
                    return;
                }
	    case 'M':
#ifdef TOOLKIT
		(void) XTranslateCoordinates(DISP, eventp->xkey.window,
			mane.win, eventp->xkey.x, eventp->xkey.y,
			&home_x, &home_y, &ww);	/* throw away last argument */
		home_x *= mane.shrinkfactor;
		home_y *= mane.shrinkfactor;
#else
		home_x = (eventp->xkey.x - (y_bar ? BAR_THICK : 0)
		    + mane.base_ax) * mane.shrinkfactor;
		home_y = (eventp->xkey.y - (x_bar ? BAR_THICK : 0)
		    + mane.base_ay) * mane.shrinkfactor;
#endif /* ??? */
		return;
	    case 'l':
		if (mane.base_ax <= 0) goto bad;
		scrollmane(mane.base_ax - 2 * (int) clip_w / 3, mane.base_ay);
		return;
	    case 'r':
		if (mane.base_ax >= abs_w - clip_w) goto bad;
		scrollmane(mane.base_ax + 2 * (int) clip_w / 3, mane.base_ay);
		return;
	    case 'u':
		if (mane.base_ay <= 0) goto bad;
		scrollmane(mane.base_ax, mane.base_ay - 2 * (int) clip_h / 3);
		return;
	    case 'd':
		if (mane.base_ay >= abs_h - clip_h) goto bad;
		scrollmane(mane.base_ax, mane.base_ay + 2 * (int) clip_h / 3);
		return;
	    case 'c':	/* unchecked scrollmane() */
		scrollwindow(&mane, mane.base_ax + eventp->xkey.x - clip_w/2,
		    mane.base_ay + eventp->xkey.y - clip_h/2);
		if (x_bar) paint_x_bar();
		if (y_bar) paint_y_bar();
		XWarpPointer(DISP, None, None, 0, 0, 0, 0,
		    clip_w/2 - eventp->xkey.x, clip_h/2 - eventp->xkey.y);
		return;

	    case '\030':	/* Control X */
		pointerlocate(&s_x, &s_y);
		screen_to_page(&mane,s_x, s_y, &page, &px, &py);
		xscroll_pages = !(xscroll_pages);
		if (yscroll_pages) yscroll_pages = 0;
		abs_w = (xscroll_pages ? total_pages: 1) * page_w;
	 	abs_h = (yscroll_pages ? total_pages: 1) * page_h;
		reconfig(); /* Adjust the scroll bars */
		page_to_screen(&mane, page, px, py, &s_x, &s_y);
		screen_to_abs(&mane, s_x, s_y, &ax, &ay);
		scrollmane(ax, ay);
		break;
	    case '\031':	/* Control Y */
		pointerlocate(&s_x, &s_y);
		screen_to_page(&mane,s_x, s_y, &page, &px, &py);
		yscroll_pages = !(yscroll_pages);
		if (xscroll_pages) xscroll_pages = 0;
		abs_w = (xscroll_pages ? total_pages: 1) * page_w;
	 	abs_h = (yscroll_pages ? total_pages: 1) * page_h;
		reconfig(); /* Adjust the scroll bars */
		page_to_screen(&mane, page, px, py, &s_x, &s_y);
		screen_to_abs(&mane, s_x, s_y, &ax, &ay);
		scrollmane(ax, ay);
		break;
	    case '\020':	/* Control P */
		Printf("Unit = %d, bitord = %d, byteord = %d\n",
		    BitmapUnit(DISP), BitmapBitOrder(DISP),
		    ImageByteOrder(DISP));
		return;
	    case 's':
		if (!arg0) {
		    int temp;
		    number0 = ROUNDUP(unshrunk_page_w, clip_w - 2);
		    temp = ROUNDUP(unshrunk_page_h, clip_h - 2);
		    if (number0 < temp) number0 = temp;
		}
		if (number0 <= 0) goto bad;
		if (number0 == mane.shrinkfactor) return;
		mane.shrinkfactor = number0;
		init_page();
		if (number0 != 1 && number0 != bak_shrink) {
		    bak_shrink = number0;
#ifdef	GREY
		    if (use_grey) init_pix(False);
#endif
		    reset_fonts();
		}
		reconfig();
		home(False);
		break;
	    case 'S':
		if (!arg0) goto bad;
		if (number0 < 0) goto bad;
		if (number0 == density) return;
		density = number0;
		reset_fonts();
		if (mane.shrinkfactor == 1) return;
		break;
#ifdef	GREY
	    case 'G':
		use_grey = (arg0 ? number0 : !use_grey);
		if (use_grey) init_pix(False);
		reset_fonts();
		break;
#endif

#if	PS
	    case 'v':
		if (!arg0 || resource._postscript != !number0) {
		    resource._postscript = !resource._postscript;
		    psp.toggle();
		}
		break;
#endif

	    case 'R':
		/* reread DVI file */
		--dvi_time;	/* then it will notice a change */
		break;
	    case 'h': /* Somebody looking for help, perhaps? */
	    case 'H':
	    case '?':
		system("xterm -e man xhdvi &"); /* Bring up man pages! */
		break;
	    default:
		break;
	}
	if (0 <= next_page && next_page < total_pages) {
	    if (current_page != next_page) {
		if (xscroll_pages) {
			scrollmane(mane.base_ax +
				(next_page - current_page)*page_w, mane.base_ay);
		} else if (yscroll_pages) {
			scrollmane(mane.base_ax, mane.base_ay +
				(next_page - current_page)*page_h);
		}
		current_page = next_page;
		hush_spec_now = hush_spec;
		if (!resource.keep_flag) home(False);
	    }
	    canit = True;
	    XFlush(DISP);
	    return;	/* don't use longjmp here:  it might be called from
			 * within the toolkit, and we don't want to longjmp out
			 * of Xt routines. */
	}
	bad:  XBell(DISP, 10);
}

/* APS added following: */
/* Cause page to be centered at least so pixel x,y is on page,
	and move mouse to this position */
centerpage(page, px, py)
int page, px, py;
{
	int s_x, s_y, ax, ay;

	page_to_screen(&mane, page, px, py, &s_x, &s_y);
	screen_to_abs(&mane, s_x, s_y, &ax, &ay);
	scrollmane(ax - clip_w/2, ay - clip_h/2); /* Center page on x,y if possible */
#ifdef WARPPOINTER
	/* and then move the mouse to the x,y location: */
	if (!pointerlocate(&s_x, &s_y)) {
		/* Pointer not on screen */
		return;
	}
	XWarpPointer(DISP, None, None, 0, 0, 0, 0,
	    ax - s_x - mane.base_ax, ay - s_y - mane.base_ay);
#endif /* WARPPOINTER */
	return;
}

#define	TRSIZE	100

void htex_can_it()
{
	canit = True;
	XFlush(DISP);
}

void
read_events(wait)
	wide_bool	wait;
{
	char	ch;
	Boolean	arg0;
	int	number0;
	XEvent	event;
	char	trbuf[TRSIZE];
#ifndef TOOLKIT
	int	s_x, s_y;
#endif
	char	*string;
	int	nbytes;

	alt_canit = False;
	for (;;) {
	    ch = '\0';
	    event_counter = event_freq;
	    /*
	     * The above line clears the flag indicating that an event is
	     * pending.  So if an event comes in right now, the flag will be
	     * set again needlessly, but we just end up making an extra call.
	     * Also, be careful about destroying the magnifying glass while
	     * writing it.
	     */
	    if (!XtPending() && (!wait || canit || mane.min_sx < MAXDIM ||
		    alt.min_sx < MAXDIM || mag_moved)) {
		if (!wait && (canit | alt_canit)) {
#if	PS
		    psp.interrupt();
#endif
		    if (allow_can) longjmp(canit_env, 1);
		}
		return;
	    }
#ifdef	TOOLKIT
	    XtNextEvent(&event);
	    if (resized) get_geom();
	    if (event.xany.window == alt.win && event.type == Expose) {
		handle_exp((Widget) NULL, (XtPointer) &alt, &event,
		    (Boolean *) NULL);
		continue;
	    }
	    if ((event.xany.window != mane.win) || (event.type != KeyPress)) {
		(void) XtDispatchEvent(&event);
		continue;
	    } /* Only track keypresses in the draw window */
	    string = trbuf;
	    nbytes = XLookupString(&event.xkey, string, TRSIZE, (KeySym *) NULL,
		(XComposeStatus *) NULL);
	    if (nbytes > 1) ch = '?';
	    if (nbytes != 0) ch = *string;
#else	/* !TOOLKIT */

	    XNextEvent(DISP, &event);
	    if (event.xany.window == mane.win || event.xany.window == alt.win) {
		struct WindowRec *wr = &mane;

		if (event.xany.window == alt.win) {
		    wr = &alt;
		    /* check in case we already destroyed the window */
		    if (alt_stat < 0) { /* destroy upon exposure */
			alt_stat = 0;
			handle_release();
			continue;
		    }
		    else
			alt_stat = 0;
		}
		switch (event.type) {
		case GraphicsExpose:
		case Expose:
		    expose(wr, event.xexpose.x, event.xexpose.y,
			event.xexpose.width, event.xexpose.height);
		    break;

		case MotionNotify:
/* APS Only using motion hints, so get the last pointer location: */
		    while (XCheckMaskEvent(DISP, PointerMotionMask, &event));
		    if (!pointerlocate(&s_x, &s_y)) break;
		    if ((keys_buttons&(Button1Mask|Button2Mask|Button3Mask|
					Button4Mask|Button5Mask)) == 0) {
	/* None of the buttons was down, so this isn't a mag motion event: */
			screen_to_page(&mane,s_x,s_y,&page,&px,&py);
			htex_displayanchor(page, px, py);
		    } else {
			new_mag_sx = s_x;
			new_mag_sy = s_y;
			mag_moved = (new_mag_sx != mag_sx || new_mag_sy != mag_sy);
		    }
		    break;

		case ButtonPress:
		    handle_button(&event.xbutton);
		    break;

		case ButtonRelease:
		    handle_release();
		    break;
		}	/* end switch */
	    }	/* end if window == {mane,alt}.win */

	    else if (event.xany.window == x_bar) {
		if (event.type == Expose)
		    XFillRectangle(DISP, x_bar, ruleGC,
			sx_bgn, 1, sx_end - sx_bgn, BAR_WID);
		else if (event.type == MotionNotify)
		    scrollmane(event.xmotion.x * abs_w / clip_w,
			mane.base_ay);
		else switch (event.xbutton.button)
		{
		    case 1:
			scrollmane(mane.base_ax + event.xbutton.x, mane.base_ay);
			break;
		    case 2:
			scrollmane(event.xbutton.x * abs_w / clip_w,
			    mane.base_ay);
			break;
		    case 3:
			scrollmane(mane.base_ax - event.xbutton.x, mane.base_ay);
		}
	    }

	    else if (event.xany.window == y_bar) {
		if (event.type == Expose)
		    XFillRectangle(DISP, y_bar, ruleGC,
			1, sy_bgn, BAR_WID, sy_end - sy_bgn);
		else if (event.type == MotionNotify)
		    scrollmane(mane.base_ax,
			event.xmotion.y * abs_h / clip_h);
		else switch (event.xbutton.button)
		{
		    case 1:
			scrollmane(mane.base_ax, mane.base_ay + event.xbutton.y);
			break;
		    case 2:
			scrollmane(mane.base_ax,
			    event.xbutton.y * abs_h / clip_h);
			break;
		    case 3:
			scrollmane(mane.base_ax, mane.base_ay - event.xbutton.y);
		}
	    }
	    else if (event.xany.window == top_level)
		switch (event.type) {
		case ConfigureNotify:
		    if (event.xany.window == top_level &&
			(event.xconfigure.width != clip_w ||
			event.xconfigure.height != clip_h)) {
			    register Window old_mane_win = mane.win;

			    clip_w = event.xconfigure.width;
			    clip_h = event.xconfigure.height;
			    reconfig();
			    if (old_mane_win == (Window) 0) home(False);
		    }
		    break;

		case MapNotify:		/* if running w/o WM */
		    if (mane.win == (Window) 0) {
			reconfig();
			home(False);
		    }
		    break;

		case KeyPress:
		    string = trbuf;
		    nbytes = XLookupString(&event.xkey, string, TRSIZE, NULL,
			NULL);
		    if (nbytes != 0) ch = *string;
		    if (nbytes > 1) ch = '?';
		    break;
		}
#endif	/* TOOLKIT */
	    if (ch == '\0') continue;
	    if (ch >= '0' && ch <= '9') {
		has_arg = True;
		number = number * 10 + sign * (ch - '0');
		continue;
	    }
	    else if (ch == '-') {
		has_arg = True;
		sign = -1;
		number = 0;
		continue;
	    }
	    number0 = number;
	    number = 0;
	    sign = 1;
	    arg0 = has_arg;
	    has_arg = False;
	    keystroke(ch, number0, arg0, &event);
	}
}

static	void
redraw(windowrec)
	struct WindowRec *windowrec;
{
	int px, py;

	currwin = *windowrec;
	min_x = currwin.min_sx + currwin.base_ax; /* Absolute x's and y's */
	min_y = currwin.min_sy + currwin.base_ay;
	max_x = currwin.max_sx + currwin.base_ax;
	max_y = currwin.max_sy + currwin.base_ay;
	can_exposures(windowrec);
	screen_to_page(windowrec, currwin.min_sx, currwin.min_sy,
						 &mincurpage, &px, &py);
	screen_to_page(windowrec, currwin.max_sx, currwin.max_sy,
						 &maxcurpage, &px, &py);
	if (debug & DBG_EVENT)
	    Printf("Redraw %d x %d at (%d, %d) (base=%d,%d)\n", max_x - min_x,
		max_y - min_y, min_x, min_y, currwin.base_ax, currwin.base_ay);
	XDefineCursor(DISP, mane.win, redraw_cursor);
	XFlush(DISP);
	if (setjmp(dvi_env)) {
	    XClearWindow(DISP, mane.win);
	    XDrawString(DISP, mane.win, foreGC,
		5, 5 + X11HEIGHT,
		dvi_oops_msg, strlen(dvi_oops_msg));
	    if (dvi_file) {
		Fclose(dvi_file);
		dvi_file = NULL;
	    }
	}
	else {
	    draw_page();
	    hush_spec_now = True;
	}
}

void
redraw_page()
{
	int px, py;

	if (debug & DBG_EVENT) Fputs("Redraw page:  ", stdout);
	XClearWindow(DISP, mane.win);

	screen_to_page(&mane, 0, 0, &mincurpage, &px, &py);
	screen_to_page(&mane, clip_w, clip_h, &maxcurpage, &px, &py);
	if (backing_store != NotUseful) {
	    mane.min_sx = mane.min_sy = 0;
	    mane.max_sx = page_w;
	    mane.max_sy = page_h;
	}
	else {
	    mane.min_sx = 0;
	    mane.max_sx = clip_w;
	    mane.min_sy = 0;
	    mane.max_sy = clip_h;
	}
	redraw(&mane);
}

/*
 *	Interrupt system for receiving events.  The program sets a flag
 *	whenever an event comes in, so that at the proper time (i.e., when
 *	reading a new dvi item), we can check incoming events to see if we
 *	still want to go on printing this page.  This way, one can stop
 *	displaying a page if it is about to be erased anyway.  We try to read
 *	as many events as possible before doing anything and base the next
 *	action on all events read.
 *	Note that the Xlib and Xt routines are not reentrant, so the most we
 *	can do is set a flag in the interrupt routine and check it later.
 *	Also, sometimes the interrupts are not generated (some systems only
 *	guarantee that SIGIO is generated for terminal files, and on the system
 *	I use, the interrupts are not generated if I use "(xdvi foo &)" instead
 *	of "xdvi foo").  Therefore, there is also a mechanism to check the
 *	event queue every 70 drawing operations or so.  This mechanism is
 *	disabled if it turns out that the interrupts do work.
 *	For a fuller discussion of some of the above, see xlife in
 *	comp.sources.x.
 */

static	void
can_exposures(windowrec)
	struct WindowRec *windowrec;
{
	windowrec->min_sx = windowrec->min_sy = MAXDIM;
	windowrec->max_sx = windowrec->max_sy = 0;
}

#if	HAS_SIGIO
/* ARGSUSED */
static	void
handle_intr(signo)
	int	signo;
{
	event_counter = 1;
	event_freq = -1;	/* forget Plan B */
}

static	void
enable_intr() {
	int	socket	= ConnectionNumber(DISP);

	(void) signal(SIGIO, handle_intr);
	(void) fcntl(socket, F_SETOWN, getpid());
	(void) fcntl(socket, F_SETFL, fcntl(socket, F_GETFL, 0) | FASYNC);
}
#endif	/* HAS_SIGIO */

void
do_pages()
{
	if (debug & DBG_BATCH) {
#ifdef	TOOLKIT
	    while (mane.min_sx == MAXDIM) read_events(True);
#else	/* !TOOLKIT */
	    while (mane.min_sx == MAXDIM)
		if (setjmp(canit_env)) break;
		else read_events(True);
#endif	/* TOOLKIT */
	    for (current_page = 0; current_page < total_pages; ++current_page) {
#ifdef	__convex__
		/* convex C turns off optimization for the entire function
		   if setjmp return value is discarded.*/
		if (setjmp(canit_env))	/*optimize me*/;
#else
		(void) setjmp(canit_env);
#endif
		canit = False;
		redraw_page();
	    }
	}
	else {	/* normal operation */
#if	HAS_SIGIO
	    enable_intr();
#endif
#ifdef	__convex__
	    /* convex C turns off optimization for the entire function
	       if setjmp return value is discarded.*/
	    if (setjmp(canit_env))	/*optimize me*/;
#else
	    (void) setjmp(canit_env);
#endif
	    for (;;) {
		if (mane.win != (Window) 0)
		    XDefineCursor(DISP, mane.win, ready_cursor);
		read_events(True);
		if (canit) {
		    canit = False;
		    can_exposures(&mane);
		    can_exposures(&alt);
		    redraw_page();
		}
		else if (mag_moved) {
		    if (alt.win == (Window) 0) mag_moved = False;
		    else if (abs(new_mag_sx - mag_sx) >
			2 * abs(new_mag_sy - mag_sy))
			    movemag(new_mag_sx, mag_sy);
		    else if (abs(new_mag_sy - mag_sy) >
			2 * abs(new_mag_sx - mag_sx))
			    movemag(mag_sx, new_mag_sy);
		    else movemag(new_mag_sx, new_mag_sy);
		}
		else if (alt.min_sx < MAXDIM) redraw(&alt);
		else if (mane.min_sx < MAXDIM) redraw(&mane);
		XFlush(DISP);
	    }
	}
}

invokedviviewer(filename, aname, height)
char *filename;
char *aname;
int height;
{
	char buf[1024];

	if (filename == NULL) return;
	if (height < 50) height = 50;
	height += 50;

	XtGetValues(draw_widget, arg_wh, XtNumber(arg_wh));

	if (aname == NULL) {
	    sprintf(buf, "%s %s -geometry %dx%d &", fullprog, filename,
			window_w, window_h); /* Same width and height */
	} else {
	    sprintf(buf, "%s %s#%s -geometry %dx%d &", fullprog, filename, aname,
			window_w, height); /* Same width, height of 100 */
	}
	system(buf);
}
