/**
 * A DVector represents a vector in R2; each coordinate
 * is a double.
 */

class DVector {
  double[] c = new double[2];

  /**
   * Create a new DVector with given coordinates.
   */
  public DVector(double x, double y) {
    this.c[0] = x;
    this.c[1] = y;
  }

  /**
   * Create a new DVector with unspecified coordinates (actually I
   * think the coords are initialized to 0, but don't count on this).
   */
  public DVector() {
  }

  /**
   * Convert a DVector to a String
   */
  public String toString() {
    return "{" + c[0] + "," + c[1] + "}";
  }

  /**
   * Return the length (magnitude) of this DVector (sqrt(x^2 + y^2))
   */
  public double length() {
    return Math.sqrt( this.c[0]*this.c[0] + this.c[1]*this.c[1] );
  }

  /**
   * Multiply this vector by -1.
   */
  public void negate() {
    this.c[0] = -this.c[0];
    this.c[1] = -this.c[1];
  }

  /**
   * Multiply this vector by the scalar s.
   */
  public void scale(double s) {
    this.c[0] *= s;
    this.c[1] *= s;
  }

  /**
   * Return the dot product of this vector with another one.
   */
  public double dot(DVector v) {
    return this.c[0]*v.c[0] + this.c[1]*v.c[1];
  }

  /**
   * Return a copy of this vector.
   */
  public DVector copy() {
    return new DVector(this.c[0], this.c[1]);
  }

  /**
   * Construct an array of DVectors; the array method takes
   * 2,4,6, or 8 doubles, and returns an array of 1,2,3, or
   * 4 DVectors, respectively.
   */
  public static DVector[] array(double x0, double y0) {
    DVector[] v = new DVector[1];
    v[0] = new DVector(x0, y0);
    return v;
  }
  public static DVector[] array(double x0, double y0,
			       double x1, double y1) {
    DVector[] v = new DVector[2];
    v[0] = new DVector(x0, y0);
    v[1] = new DVector(x1, y1);
    return v;
  }
  public static DVector[] array(double x0, double y0,
			       double x1, double y1,
			       double x2, double y2) {
    DVector[] v = new DVector[3];
    v[0] = new DVector(x0, y0);
    v[1] = new DVector(x1, y1);
    v[2] = new DVector(x2, y2);
    return v;
  }
  public static DVector[] array(double x0, double y0,
			       double x1, double y1,
			       double x2, double y2,
			       double x3, double y3) {
    DVector[] v = new DVector[4];
    v[0] = new DVector(x0, y0);
    v[1] = new DVector(x1, y1);
    v[2] = new DVector(x2, y2);
    v[3] = new DVector(x3, y3);
    return v;
  }

  /**
   * Convert an array of DVectors to a string.
   */
  public static String aString(DVector[] v) {
    StringBuffer buf = new StringBuffer();
    buf.append("[");
    if (v != null) {
      for (int i=0; i<v.length; ++i) {
	if (i>0) { buf.append(","); }
	buf.append(v[i]);
      }
    }
    buf.append("]");
    return buf.toString();
  }

}

/**
 * A Segment is simply a collection of two DVectors; it's the
 * basic unit of drawing in Kali.
 */
class Segment {

  /**
   * The array of 2 DVectors, which are the endpoints of the
   * segment.
   */
  DVector[] p = new DVector[2];

  /**
   * Create a new segment from 4 doubles
   */
  public Segment(double x0, double y0, double x1, double y1) {
    this.p[0] = new DVector(x0,y0);
    this.p[1] = new DVector(x1,y1);
  }

  /**
   * Create a new segment from 2 DVectors.
   */
  public Segment(DVector p0, DVector p1) {
    this.p[0] = p0;
    this.p[1] = p1;
  }

  /**
   * Return a new segment which is a copy of this one.
   */
  public Segment copy() {
    return new Segment(p[0].copy(), p[1].copy());
  }

  /**
   * Swap the two endpoints of this segment.
   */
  public void reverse() {
    DVector temp = p[0];
    p[0] = p[1];
    p[1] = temp;
  }

  /**
   * Convert this segment to a string
   */
  public String toString() {
    return "{" + p[0] + "," + p[1] + "}";
  }

  /**
   * Construct an array of Segments; the array method takes
   * 4, 8, 12, or 16 doubles, and returns an array of 1,2,3, or
   * 4 Segments, respectively.
   */
  public static Segment[] array(double x0, double y0, double x1, double y1) {
    Segment[] s = new Segment[1];
    s[0] = new Segment(x0, y0, x1, y1);
    return s;
  }
  public static Segment[] array(double x0, double y0, double x1, double y1,
				double x2, double y2, double x3, double y3) {
    Segment[] s = new Segment[2];
    s[0] = new Segment(x0, y0, x1, y1);
    s[1] = new Segment(x2, y2, x3, y3);
    return s;
  }
  public static Segment[] array(double x0, double y0, double x1, double y1,
				double x2, double y2, double x3, double y3,
				double x4, double y4, double x5, double y5) {
    Segment[] s = new Segment[3];
    s[0] = new Segment(x0, y0, x1, y1);
    s[1] = new Segment(x2, y2, x3, y3);
    s[2] = new Segment(x4, y4, x5, y5);
    return s;
  }
  public static Segment[] array(double x0, double y0, double x1, double y1,
				double x2, double y2, double x3, double y3,
				double x4, double y4, double x5, double y5,
				double x6, double y6, double x7, double y7) {
    Segment[] s = new Segment[4];
    s[0] = new Segment(x0, y0, x1, y1);
    s[1] = new Segment(x2, y2, x3, y3);
    s[2] = new Segment(x4, y4, x5, y5);
    s[3] = new Segment(x6, y6, x7, y7);
    return s;
  }

  /**
   * Convert an array of Segments to a string
   */
  public static String aString(Segment[] s) {
    StringBuffer buf = new StringBuffer();
    buf.append("[");
    if (s != null) {
      for (int i=0; i<s.length; ++i) {
	if (i>0) { buf.append(","); }
	buf.append(s[i]);
      }
    }
    buf.append("]");
    return buf.toString();
  }

}

/**
 * A DMatrix is a 2x2 matrix; each entry is a double.
 */
class DMatrix {
  double[][] c;

  /**
   * Create a new DMatrix with the given entries.
   */
  public DMatrix(double x00,double x01,
		 double x10,double x11) {
    c = new double[2][2];
    c[0][0] = x00;
    c[0][1] = x01;
    c[1][0] = x10;
    c[1][1] = x11;
  }

  /**
   * Create a new matrix with unspecified entries.
   */
  public DMatrix() {
    c = new double[2][2];
    c[0][0] = 1.0;
    c[1][1] = 1.0;
  }
  
  /**
   * Multiply this matrix by another one. This is RIGHT multiplication:
   * M.times(N) returns M * N.
   */
  public DMatrix times(DMatrix N) {
    DMatrix result = new DMatrix();
    for (int i=0; i<2; ++i) {
      for (int j=0; j<2; ++j) {
	result.c[i][j] = this.c[i][0]*N.c[0][j] + this.c[i][1]*N.c[1][j];
      }
    }
    return result;
  }

  /**
   * Multiply this matrix by a vector on the RIGHT:
   * M.times(v) returns M * v.
   */
  public DVector times(DVector v) {
    DVector result = new DVector();
    result.c[0] = this.c[0][0]*v.c[0] + this.c[0][1]*v.c[1];
    result.c[1] = this.c[1][0]*v.c[0] + this.c[1][1]*v.c[1];
    return result;
  }

  /**
   * Return the determinant of this matrix.
   */
  public double det() {
    return c[0][0]*c[1][1] - c[0][1]*c[1][0];
  }

  /**
   * Return the inverse of this matrix.
   */
  public DMatrix inverse() {
    double det = det();
    DMatrix result = new DMatrix( c[1][1]/det, -c[0][1]/det,
				 -c[1][0]/det,  c[0][0]/det );
    return result;
  }

  /**
   * Convert this matrix to a String.
   */
  public String toString() {
    return "{{" + c[0][0] + "," + c[0][1] + "},{" + c[1][0] + "," + c[1][1] + "}}";
  }

}
